package projekt.enigma.model;

import projekt.enigma.database.DatenbankSqlite;

import java.time.LocalDate;
import java.time.ZoneId;

/**
 * Das Codebuch war eines der essentiel wichtigen Sachen bei der Enigma.
 * Jeder Enigma gehörte auch ein Codebuch bei, welches notwenidg war um die Nachrichten zu ver/entschlüsseln.
 * Im Codebuch standen für jeden Tag des Monats die notwendigen Einstellungen welche vorgenommen werden
 * mussten um die Texte zu ver/entschlüsseln.
 * Hiernach wurden die Walzen eingesetzt, die Ringe für den Rücksprung der Walze gesetzt und deren Grundeinstellung
 * vorgenommen.
 * Bei der Enigma um welche es hier geht (Enigma I) gab es zusätzlich ein Steckbrett, welches ebenfalls im Codebuch
 * festgelegt war.
 * <p>
 * Dieses Codebuch Objekt enthält die Einstellungen eines Tages, welche mittels Getter und Setter
 * verarbeitet werden können.
 * <p>
 * Zusätzlich gibt es zu Debug Zwecken eine überschriebene toString Funktion welche das Objekt als String zurück
 * geben kann.
 */
public class Codebuch {
	/**
	 * TODO: Dokumentieren
	 */
	private int tag;
	private int[] walzenlage;
	private int[] ringstellung;
	private char[][] steckverbindung;

	/**
	 * Im Konstruktor werden die Standart Werte gesetzt
	 * <p>
	 * walzenlage : int[3] :            Hier wird die Position der drei Walzen festgelegt.
	 * ringstellung : char[3] :         Jede Walze hat einen Ring. Der Index steht für die Walze und der Value für
	 * die Position des Ringes.
	 * steckverbindung : char[10][] :   Zehn Kabel können in das Steckbrett gesteckt werden. In diesem Array werden
	 * die Kabel mit ihren jeweiligen Buchsen hinterlegt.
	 */
	public Codebuch() {
		this.walzenlage = new int[3];
		this.ringstellung = new int[3];
		this.steckverbindung = new char[10][];

		for (int i = 0; i < 10; i++) {
			this.steckverbindung[i] = new char[2];
			this.steckverbindung[i][0] = 'A';
			this.steckverbindung[i][1] = 'A';
		}
	}

	/**
	 * TODO: Dokumentieren
	 * @param tag
	 */
	public void fetchTagesschluessel(int... tag) {
		String[] walzenlage, ringstellung, db;

		if(tag.length > 0) {
			db = new DatenbankSqlite().getCodebuch(tag[0]);
		} else {
			db = new DatenbankSqlite().getCodebuch(LocalDate.now(ZoneId.of("Europe/Berlin")).getDayOfMonth());
		}

		this.setTag(Integer.parseInt(db[0]));

		walzenlage = db[1].split(",");
		for (int i = 0; i < 3; i++) {
			this.setWalzenlage(i, Integer.parseInt(walzenlage[i]));
		}

		ringstellung = db[2].split(",");
		for (int i = 0; i < 3; i++) {
			this.setRingstellung(i, Integer.parseInt(ringstellung[i]));
		}

		this.setSteckverbindung(db[3].split(","));

	}

	/**
	 * Gibt die aktuelle Instanz des Codebuches als String zurück.
	 * Hierbei werden der Tag die Walzenlagen, Ring/Grundstellung sowie die Steckverbindungen zurück gegeben.
	 *
	 * @return String : String des gesamten Objektes
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		// Tag des aktuellen Codebuch Eintrags
		sb.append("Tag: ").append(this.getTag()).append("\n");

		// In welchem Slot steckt die Walze?
		sb.append("Walzenlage: ");
		for (int wl : this.getWalzenlage()) {
			sb.append(wl).append(",");
		}
		if (this.getWalzenlage().length > 0) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		// Die Ringstellung an den Walzen
		sb.append("Ringstellung: ");
		for (int rs : this.getRingstellung()) {
			sb.append(rs).append(",");
		}
		if (this.getRingstellung().length > 0) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		// Das Steckbrett und wie die Kabel angeschlossen sind
		sb.append("Steckverbindung: ");
		for (char[] verbindung : this.getSteckverbindung()) {
			sb.append(verbindung[0]).append(":").append(verbindung[1]).append(",");
		}
		if (this.getSteckverbindung().length > 0) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		return sb.toString();
	}

	/**
	 * Gibt den Tag dieser Instanz zurück
	 *
	 * @return int : Tag
	 */
	public int getTag() {
		return tag;
	}

	/**
	 * Setzt den Tag dieser Instanz
	 *
	 * @param tag : int
	 */
	void setTag(int tag) {
		this.tag = tag;
	}

	/**
	 * Gibt die Walzenlage dieser Instanz zurück
	 *
	 * @return String[] : Walzenlage
	 */
	public int[] getWalzenlage() {
		return walzenlage;
	}

	/**
	 * Gibt die Ringstellung dieser Instanz zurück
	 *
	 * @return String[] : Ringstellung
	 */
	public int[] getRingstellung() {
		return ringstellung;
	}

	/**
	 * Gibt die Steckverbindung dieser Instanz zurück
	 *
	 * @return String[] : Steckverbindung
	 */
	// TODO
	public char[][] getSteckverbindung() {
		return this.steckverbindung;
	}

	/**
	 * Setzt die Steckverbindung dieser Instanz
	 *
	 * @param kabel : String[] :  Die Steckverbindungen die gesteckt werden sollen.
	 *              Jeder Eintrag im Array ist ein String mit zwei Zeichen.
	 *              Diese werden in zwei char zerlegt und anschließend im globalen
	 *              Array gespeichert.
	 */
	private void setSteckverbindung(String[] kabel) {
		int counter = 0;

		for (String stecker : kabel) {
			char[] verbindung = new char[2];
			verbindung[0] = stecker.charAt(0);
			verbindung[1] = stecker.charAt(1);
			this.steckverbindung[counter++] = verbindung;
		}
	}

	/**
	 * Setzt die Walzenlage dieser Instanz
	 *
	 * @param position   : int : Position der Walze
	 * @param walzenlage : int : Welche Walze wurde eingesetzt
	 */
	private void setWalzenlage(int position, int walzenlage) {
		this.walzenlage[position] = walzenlage;
	}

	/**
	 * Setzt die Ringstellung dieser Instanz
	 *
	 * @param position     : int : Auf welche Walze wird der Ring aufgesetzt?
	 * @param ringstellung : char : An dieser Position soll die nächste Walze weiter gedreht werden.
	 */
	private void setRingstellung(int position, int ringstellung) {
		this.ringstellung[position] = ringstellung;
	}
}
