package Enigma.model;

/**
 * Das Codebuch war eines der essentiel wichtigen Sachen bei der Enigma.
 * Jeder Enigma gehörte auch ein Codebuch bei, welches notwenidg war um die Nachrichten zu ver/entschlüsseln.
 * Im Codebuch standen für jeden Tag des Monats die notwendigen Einstellungen welche vorgenommen werden
 * mussten um die Texte zu ver/entschlüsseln.
 * Hiernach wurden die Walzen eingesetzt, die Ringe für den Rücksprung der Walze gesetzt und deren Grundeinstellung
 * vorgenommen.
 * Bei der Enigma um welche es hier geht (Enigma I) gab es zusätzlich ein Steckbrett, welches ebenfalls im Codebuch
 * festgelegt war.
 * <p>
 * Dieses Codebuch Objekt enthält die Einstellungen eines Tages, welche mittels Getter und Setter
 * verarbeitet werden können.
 * <p>
 * Zusätzlich gibt es zu Debug Zwecken eine überschriebene toString Funktion welche das Objekt als String zurück
 * geben kann.
 */
public class Codebuch {
	private int tag;
	private int[] walzenlage;
	private int[] ringstellung;
	private char[][] steckverbindung;

	/**
	 * Im Konstruktor werden die Standart Werte gesetzt
	 * <p>
	 * walzenlage : int[3] :            Hier wird die Position der drei Walzen festgelegt.
	 * ringstellung : char[3] :         Jede Walze hat einen Ring. Der Index steht für die Walze und der Value für
	 * die Position des Ringes.
	 * steckverbindung : char[10][] :   Zehn Kabel können in das Steckbrett gesteckt werden. In diesem Array werden
	 * die Kabel mit ihren jeweiligen Buchsen hinterlegt.
	 */
	public Codebuch() {
		this.walzenlage = new int[3];
		this.ringstellung = new int[3];
		this.steckverbindung = new char[10][];

		for (int i = 0; i < 10; i++) {
			this.steckverbindung[i] = new char[2];
			this.steckverbindung[i][0] = 'A';
			this.steckverbindung[i][1] = 'A';
		}
	}

	/**
	 * Gibt die aktuelle Instanz des Codebuches als String zurück.
	 * Hierbei werden der Tag die Walzenlagen, Ring/Grundstellung sowie die Steckverbindungen zurück gegeben.
	 *
	 * @return String : String des gesamten Objektes
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		boolean wlLoop = false;
		boolean rsLoop = false;
		boolean svLoop = false;

		sb.append("Tag: ").append(this.getTag()).append("\n");

		sb.append("Walzenlage: ");
		for (int wl : this.getWalzenlage()) {
			sb.append(wl).append(",");
			wlLoop = true;
		}
		if (wlLoop) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		sb.append("Ringstellung: ");
		for (int rs : this.getRingstellung()) {
			sb.append(rs).append(",");
			rsLoop = true;
		}
		if (rsLoop) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		sb.append("Steckverbindung: ");
		for (char[] verbindung : this.getSteckverbindung()) {
			sb.append(verbindung[0]).append(":").append(verbindung[1]).append(",");
			svLoop = true;
		}
		if (svLoop) {
			sb.setLength(sb.length() - 1);
		}
		sb.append("\n");

		return sb.toString();
	}

	/**
	 * Gibt den Tag dieser Instanz zurück
	 *
	 * @return int : Tag
	 */
	public int getTag() {
		return tag;
	}

	/**
	 * Setzt den Tag dieser Instanz
	 *
	 * @param tag : int
	 */
	public void setTag(int tag) {
		this.tag = tag;
	}

	/**
	 * Gibt die Walzenlage dieser Instanz zurück
	 *
	 * @return String[] : Walzenlage
	 */
	public int[] getWalzenlage() {
		return walzenlage;
	}

	/**
	 * Gibt die Ringstellung dieser Instanz zurück
	 *
	 * @return String[] : Ringstellung
	 */
	public int[] getRingstellung() {
		return ringstellung;
	}

	/**
	 * Gibt die Steckverbindung dieser Instanz zurück
	 *
	 * @return String[] : Steckverbindung
	 */
	public char[][] getSteckverbindung() {
		return this.steckverbindung;
	}

	/**
	 * Setzt die Steckverbindung dieser Instanz
	 *
	 * @param kabel : String[] :  Die Steckverbindungen die gesteckt werden sollen.
	 *                        Jeder Eintrag im Array ist ein String mit zwei Zeichen.
	 *                        Diese werden in zwei char zerlegt und anschließend im globalen
	 *                        Array gespeichert.
	 */
	public void setSteckverbindung(String[] kabel) {
		int counter = 0;

		for (String stecker : kabel) {
			char[] verbindung = new char[2];
			verbindung[0] = stecker.charAt(0);
			verbindung[1] = stecker.charAt(1);
			this.steckverbindung[counter++] = verbindung;
		}
	}

	/**
	 * Setzt die Walzenlage dieser Instanz
	 *
	 * @param position   : int : Position der Walze
	 * @param walzenlage : int : Welche Walze wurde eingesetzt
	 */
	public void setWalzenlage(int position, int walzenlage) {
		this.walzenlage[position] = walzenlage;
	}

	/**
	 * Setzt die Ringstellung dieser Instanz
	 *
	 * @param position     : int : Auf welche Walze wird der Ring aufgesetzt?
	 * @param ringstellung : char : An dieser Position soll die nächste Walze weiter gedreht werden.
	 */
	public void setRingstellung(int position, int ringstellung) {
		this.ringstellung[position] = ringstellung;
	}
}
