package projekt.enigma.model;

import java.util.*;
import java.util.Map.Entry;

/**
 * Diese Klasse definiert die Zugehörigkeit zwischen den Buchstaben A-Z und dem zugehörigen Morsecode.
 * Die Funktionen sollen Texte von und zu Morsecode konvertieren.
 */
public class Morsecode {
	// Anlegen der codeMap als TreeMap
	private TreeMap<String, String> codeMap = new TreeMap<>();

	/**
	 * Im Konstruktor werden alle verfügbaren Buchstaben (A-Z) und deren zugehörigen
	 * Morsecodes, der codeMap hinzugefügt.
	 */
	public Morsecode() {
		this.codeMap.put("A", ".-");
		this.codeMap.put("B", "-...");
		this.codeMap.put("C", "-.-.");
		this.codeMap.put("D", "-..");
		this.codeMap.put("E", ".");
		this.codeMap.put("F", "..-.");
		this.codeMap.put("G", "--.");
		this.codeMap.put("H", "....");
		this.codeMap.put("I", "..");
		this.codeMap.put("J", ".---");
		this.codeMap.put("K", "-.-");
		this.codeMap.put("L", ".-..");
		this.codeMap.put("M", "--");
		this.codeMap.put("N", "-.");
		this.codeMap.put("O", "---");
		this.codeMap.put("P", ".--.");
		this.codeMap.put("Q", "--.-");
		this.codeMap.put("R", ".-.");
		this.codeMap.put("S", "...");
		this.codeMap.put("T", "-");
		this.codeMap.put("U", "..-");
		this.codeMap.put("V", "...-");
		this.codeMap.put("W", ".--");
		this.codeMap.put("X", "-..-");
		this.codeMap.put("Y", "-.--");
		this.codeMap.put("Z", "--..");
		this.codeMap.put(" ", "-...-");
		this.codeMap.put("1", ".----");
		this.codeMap.put("2", ".----");
		this.codeMap.put("3", "...--");
		this.codeMap.put("4", "....-");
		this.codeMap.put("5", ".....");
		this.codeMap.put("6", "-....");
		this.codeMap.put("7", "--...");
		this.codeMap.put("8", "---..");
		this.codeMap.put("9", "----.");
		this.codeMap.put("0", "-----");
	}

	/**
	 * Diese Funktion überprüft ob das übergebene Buchstabe in der codeMap enthalten ist.
	 * Wenn es enthalten ist, wird der zugehörige Morsecode zurück gegeben, ansonsten null
	 *
	 * @param buchstabe : char : Zu übersetzender Buchstabe
	 * @return : String : Das zugehöriger Morsecode aus der codeMap
	 */
	private String getMorsecode(char buchstabe) {
		// Testen ob das Char in der codeMap enthalten ist. Wenn ja, raussuchen und zurück geben
		return this.codeMap.getOrDefault(String.valueOf(buchstabe).toUpperCase(), null);
	}

	/**
	 * Der übergebene String wird in seine einzelnen Zeichen (char) zerlegt und an die Funktion getMorsecode
	 * übergeben und der zugehörige Morsecode raus gesucht.
	 * Mittels eines StringBuilders werden die Ergebnisse zu einem Rückgabe String zusammen gesetzt.
	 * Wenn kein zugehöriger Code in der codeMap gefunden wurde, wird das Zeichen ignoriert da es bei der
	 * Enigma nicht vorhanden war.
	 *
	 * @param input : String : Zu konventierender Text
	 * @return String : Die in Morsecode konvertierte Text
	 */
	public String convertBuchstabeToMorsecode(String input) {
		StringBuilder sb = new StringBuilder();
		String morsecode;

		// Durch alle char im übergebenen String loopen
		for(int i = 0; i < input.length(); i++) {

			// Wenn der char in der codeMap enthalten ist, an den Rückgabe String anheften, ansonsten nichts ignorieren.
			morsecode = this.getMorsecode(input.charAt(i));
			if(morsecode != null) {
				sb.append(morsecode).append(' ');
			}
		}

		// Gibt den konvertierten String zurück
		return sb.toString();
	}

	/**
	 * Diese Funktion überprüft ob der übergebene Morsecode in der codeMap enthalten ist.
	 * Wenn er enthalten ist, wird der zugehörige Buchstabe zurück gegeben, ansonsten null
	 *
	 * @param morsecode : String : Der zu übersetzende Morsecode
	 * @return : String : Das zugehöriger Buchstaben aus der this.codeMap
	 */
	private String getBuchstabe(String morsecode) {
		// Initialisieren des Rückgabe Strings
		String result = null;

		// Loopt duch die Codemap und erstellt daraus ein Set
		for(Entry<String, String> entry : this.codeMap.entrySet()) {
			// Wenn Value dem Morsecode entspricht, haben wir unseren Buchstaben gefunden und setzen ihn als Result
			if(entry.getValue().equals(morsecode)) {
				result = entry.getKey();
			}
		}

		return result;
	}

	/**
	 * Der übergebene String wird in einen String Array zerlegt. Als Trennzeichen wird das Leerzeichen genutzt.
	 * Anschließend wird der Array durchlaufen und die Elemente an die Funktion getBuchstabe übergeben um den
	 * zugehörigen Buchstaben zu erhalten.
	 * Mittels eines StringBuilders werden die Ergebnisse zu einem Rückgabe String zusammen gesetzt.
	 * Wenn kein zugehöriger Buchstabe in der codeMap gefunden wurde, wird das Zeichen ignoriert da es bei der
	 * Enigma nicht vorhanden war.
	 *
	 * @param input : String : Zu konventierender Morsecode
	 * @return String : Der in Text konvertierte Morsecode
	 */
	public String convertMorsecodeToBuchstabe(String input) {
		StringBuilder sb = new StringBuilder();
		String buchstabe;

		String[] morsecodes = input.split(" ");

		// Durch alle char im übergebenen String loopen
		for(String morsecode : morsecodes) {

			// Wenn der char in der codeMap enthalten ist, an den Rückgabe String anheften, ansonsten nichts ignorieren.
			buchstabe = this.getBuchstabe(morsecode);
			if(buchstabe != null) {
				sb.append(buchstabe);
			}
		}

		// Gibt den konvertierten String zurück
		return sb.toString();
	}
}
